//-----------------------------------------------------------------------------
//
// A bare-bones skeleton based on the Microsoft DirectX SDK Tutorial 1
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Header files
//-----------------------------------------------------------------------------

#include <d3d8.h>
#include <d3dx8.h>

//-----------------------------------------------------------------------------
// Structors that should be in a header
//-----------------------------------------------------------------------------

struct MYVERTEX
{
  FLOAT x, y, z; // The transformed position
  FLOAT rhw;     // 1.0 (reciprocal of homogeneous w)
  DWORD color;   // The vertex color
};

//-----------------------------------------------------------------------------
// DX8 Information Class - a convenient, single place to store global DX vars
//-----------------------------------------------------------------------------

class DX8VARS {
public:
	LPDIRECT3D8 pD3D; // Used to create the D3DDevice
	
	LPDIRECT3DDEVICE8 pd3dDevice; // Our rendering device
	
	LPD3DXSPRITE pd3dxSprite; // Interface to Sprite routines

	IDirect3DVertexBuffer8 *pStreamData; //Vertex for ???

	IDirect3DIndexBuffer8 * pIndexBuffer; //I have no Idea

	IDirect3DTexture8 * pTexture; //Holds Texture.  Just one

	D3DCAPS8 *pd3dcaps; // Caps for the rendering device
	
	// Constructor/Destructor:
	DX8VARS() { pD3D = NULL; pd3dDevice = NULL; pd3dxSprite = NULL;
	pd3dcaps = new D3DCAPS8; }
	~DX8VARS() { if (pd3dcaps) delete pd3dcaps; }
	// If you feel very object-oriented, you could make InitD3D(), Cleanup()
	// and Render() all methods of the DX8VARS class.
}; 

//----------------------------------------------------------------------------
// Global variables
//----------------------------------------------------------------------------
DX8VARS dx8;

MYVERTEX vertices[] =
{
    { 150.0f,  50.0f, 0.5f, 1.0f, 0x000000ff}, // x, y, z, rhw, color
    { 250.0f, 250.0f, 0.5f, 1.0f, 0x000000cc},
    {  50.0f, 250.0f, 0.5f, 1.0f, 0x00cc00cc}
};

WORD indices[] = { 0, 1, 2 };

//----------------------------------------------------------------------------
// InitD3D() : Initializes Direct3D
//----------------------------------------------------------------------------

HRESULT InitD3D( HWND hWnd )
{
	// Create the D3D object, which is needed to create the D3DDevice.
	if( ( dx8.pD3D = Direct3DCreate8( D3D_SDK_VERSION ) ) == NULL )
		return E_FAIL;
	
	// Get the current desktop display mode
	D3DDISPLAYMODE d3ddm;
	if( FAILED( dx8.pD3D->GetAdapterDisplayMode( D3DADAPTER_DEFAULT,
		&d3ddm )))
		return E_FAIL;
	
	// Parameters for the D3DDevice. Most parameters are zero'd out.
	// This sets the video format to match the current desktop display.
	// Check docs for other options you can set here, such as
	// 'Windowed' which would create a window-based app (not full screen)
	D3DPRESENT_PARAMETERS d3dpp; 
	ZeroMemory( &d3dpp, sizeof(d3dpp) );
	d3dpp.SwapEffect = D3DSWAPEFFECT_DISCARD;
	d3dpp.BackBufferWidth = d3ddm.Width;
	d3dpp.BackBufferHeight = d3ddm.Height;
	d3dpp.BackBufferFormat = d3ddm.Format;
	



	// Create the Direct3D device, using the default adapter (most systems 
	// only have one, unless they have multiple graphics hardware cards
	// installed). See SDK for more details on other options here.
	if( FAILED( dx8.pD3D->CreateDevice( D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, hWnd,D3DCREATE_SOFTWARE_VERTEXPROCESSING,&d3dpp, &dx8.pd3dDevice ) ) )
		return E_FAIL;
	
	// Device state would normally be set here
	int num_elems = sizeof(vertices) / sizeof(vertices[0]);

	dx8.pd3dDevice->CreateVertexBuffer(sizeof(MYVERTEX) * num_elems, D3DUSAGE_WRITEONLY,
			D3DFVF_XYZRHW|D3DFVF_DIFFUSE, D3DPOOL_DEFAULT, &dx8.pStreamData);

	MYVERTEX *v;
	dx8.pStreamData->Lock(0, 0, (BYTE**)&v, 0);
	for(int ii = 0; ii < num_elems; ii++){
		v[ii].x     = vertices[ii].x;
		v[ii].y     = vertices[ii].y;
		v[ii].z     = vertices[ii].z;
		v[ii].rhw   = vertices[ii].rhw;
		v[ii].color = vertices[ii].color;
	}
	dx8.pStreamData->Unlock();

	num_elems = sizeof(indices) / sizeof(indices[0]);
	dx8.pd3dDevice->CreateIndexBuffer(sizeof(WORD) * num_elems, D3DUSAGE_WRITEONLY, 
		D3DFMT_INDEX16, D3DPOOL_DEFAULT, &dx8.pIndexBuffer);

	WORD *pIndex;
	dx8.pIndexBuffer->Lock(0, 0, (BYTE **)&pIndex, 0);
	for(ii = 0; ii < num_elems; ii++){
		pIndex[ii] = indices[ii];
	}
	dx8.pIndexBuffer->Unlock();

	return S_OK;
}


//----------------------------------------------------------------------------
// Cleanup() : Releases all previously initialized objects
//----------------------------------------------------------------------------
void Cleanup()
{
	if( dx8.pd3dDevice ) 
		dx8.pd3dDevice->Release();
	
	if( dx8.pD3D )
		dx8.pD3D->Release();
}


//----------------------------------------------------------------------------
// DrawScene() : Draws the contents of the scene
//----------------------------------------------------------------------------
void DrawScene()
{
	// Drawing code goes here...
	
	int num_elems = sizeof(vertices) / sizeof(vertices[0]);
	dx8.pd3dDevice->SetVertexShader(D3DFVF_XYZRHW | D3DFVF_DIFFUSE);
	dx8.pd3dDevice->SetStreamSource(0, dx8.pStreamData, sizeof(MYVERTEX));
//	dx8.pd3dDevice->DrawPrimitive(D3DPT_TRIANGLELIST,0,1);

	dx8.pd3dDevice->SetIndices(dx8.pIndexBuffer, 0);	
	dx8.pd3dDevice->DrawIndexedPrimitive(
                 D3DPT_TRIANGLELIST,
                 0,
                 sizeof(indices) / sizeof(indices[0]),
                 0,
                 sizeof(indices) / sizeof(indices[0]) / 3);

	return;
}


//----------------------------------------------------------------------------
// Render() : Prepares for drawing the scene
//----------------------------------------------------------------------------
void Render()
{
	if( dx8.pd3dDevice == NULL)
		return;
	
	// Clear the backbuffer to a black color
	dx8.pd3dDevice->Clear( 0, NULL, D3DCLEAR_TARGET, 
		D3DCOLOR_XRGB(0,0,0), 1.0f, 0 );
	
	// Begin the scene
	dx8.pd3dDevice->BeginScene();
	
	// Rendering of scene objects
	DrawScene();
	
	// End the scene
	dx8.pd3dDevice->EndScene();
	
	// Present the backbuffer contents to the display
	dx8.pd3dDevice->Present( NULL, NULL, NULL, NULL );
}


//----------------------------------------------------------------------------
// MsgProc() : The window's message handler
//----------------------------------------------------------------------------
LRESULT WINAPI MsgProc( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
	int ii;
	switch( msg ){
		case WM_KEYUP: 
			switch (wParam) { 
				case VK_ESCAPE:
					PostQuitMessage( 0 );
                    break;
            } 
            return 0;
		case WM_CHAR:
			MYVERTEX *v;
			dx8.pStreamData->Lock(0, 0, (BYTE**)&v, 0);
			for(ii = 0; ii < 3; ii++){
				v[ii].x     += 5;
				v[ii].y     += 5;
			}
			dx8.pStreamData->Unlock();
			Render();
			return 0;
		case WM_DESTROY: // If you use Windowed mode process loss of it
			PostQuitMessage( 0 );
			return 0;
		
		case WM_PAINT:
			// NOTE: In a full-screen app, you may choose to draw the screen at
			// a different point, such as using a timer message or outside of
			// the windows message loop
	        Render();
			ValidateRect( hWnd, NULL );
			return 0;
	}
	return DefWindowProc( hWnd, msg, wParam, lParam );
}


//----------------------------------------------------------------------------
// WinMain() : The application's entry point
//----------------------------------------------------------------------------
INT WINAPI WinMain( HINSTANCE hInst, HINSTANCE, LPSTR, INT )
{
	// Register the window class
	WNDCLASSEX wc = { sizeof(WNDCLASSEX), CS_CLASSDC, MsgProc, 0L, 0L, 
		GetModuleHandle(NULL), NULL, NULL, NULL, NULL,
		"Tetris Extreme", NULL };
	RegisterClassEx( &wc );
	
	// Create the application's window
	HWND hWnd = CreateWindow( "Tetris Extreme", "Tetris Extreme", 
		WS_OVERLAPPEDWINDOW, 100, 100, 300, 300,
		GetDesktopWindow(), NULL, wc.hInstance, NULL );
	
	// Initialize Direct3D
	if( SUCCEEDED( InitD3D( hWnd ) ) )
	{
		// Show the window
		ShowWindow( hWnd, SW_SHOWDEFAULT );
		UpdateWindow( hWnd ); // Forces a WM_PAINT
		
		// Enter the message loop
		MSG msg; 
		while( GetMessage( &msg, NULL, 0, 0 ) )
		{
			TranslateMessage( &msg );
			DispatchMessage( &msg );
		}
	}
	
	// Clean up everything and exit the app
	Cleanup();
	UnregisterClass( "Tetris Extreme", wc.hInstance );
	return 0;
}

